/**
* \file: AilAudioOutImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Michalik / ADIT/SW2 / jmichalik@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <memory.h>
#include <pthread.h>
#include <sys/prctl.h>
#include <limits.h>
#include <adit_logging.h>
#include <dipo_macros.h>
#include "AilAudioOut.h"
#include "AilAudioOutImpl.h"
#include "AilConfiguration.h"

#include "AudioFactory.h"
#include "AudioBackend.h"
#include "AudioTypes.h"


using namespace std;

LOG_IMPORT_CONTEXT(cply);

namespace adit { namespace carplay
{

static const char* _getChannelName(AudioChannelType inChannel)
{
    if (inChannel == AudioChannelType_Main)
        return "main";
    else if (inChannel == AudioChannelType_Alternate)
        return "alternate";
    else
        return "";
}

AilAudioOut::Impl::Impl()
{
    config = nullptr;
    source = nullptr;

    periodMilli = 0;
    periodSamples = 0;
    appendEmptyPeriods = 0;
    channel = AudioChannelType_Main;
    verboseLogging = false;
    running = false;
    std::string backendLibName("Alsa");
    backend = audio::Factory::Instance()->createBackend(backendLibName,*this);
}

AilAudioOut::Impl::~Impl()
{
    /* PRQA: Lint Message 1506: Stop is not expected to be further overridden */
    /*lint -save -e1506*/
    Stop();
    /*lint -restore*/
}

bool AilAudioOut::Impl::Initialize(const IConfiguration& inConfig, IAudioOutSource& inSource)
{
    config = &inConfig;
    source = &inSource;

    verboseLogging = AilConfiguration::Instance().IsVerboseLogging(inConfig);

    return true;
}

bool AilAudioOut::Impl::Prepare(AudioFormatStruct inFormat, AudioChannelType inChannel, const std::string& inAudioType)
{
	int32_t buffer_periods = -1; // not necessary to configure AIL
	int32_t silence_ms = -1;     // not necessary to configure AIL
	int32_t inittout_ms = -1;

	if (config == nullptr)
    {
        LOG_ERROR((cply, "AilAudioOut is not initialized"));
        return false;
    }

    channel = inChannel;
    format = inFormat;

    if (format.BitsPerChannel != 24 && format.BitsPerChannel != 16)
    {
        LOG_ERROR((cply, "AilAudioOut does not support %d bit audio", format.BitsPerChannel));
        return false;
    }

    // numbers of periods to append after end-of-stream
    appendEmptyPeriods = config->GetNumber("alsa-audio-out-append-empty-periods", 0); //ToDo: What is the purpose of this?

    if (!AilConfiguration::Instance().GetDeviceSettings(*config,
            (AilConfiguration::Channel) inChannel, inAudioType, inFormat, deviceName /* out */,
            periodMilli /* out */, buffer_periods /* out */, silence_ms /* out */, inittout_ms /* out */))
    {
        LOG_ERROR((cply, "failed to get audio device settings"));
        return false; /* ========== leaving function ========== */
    }

    audio::AudioFormat audioFormat;
    switch (format.BitsPerChannel) {
        case 16 : {
            audioFormat = S16_LE;
            break;
        }
        case 24 : {
            audioFormat = S24_LE;
            break;
        }
        default :
            LOG_ERROR((cply, "AilAudioOut does not support %d bits per channel\n", format.BitsPerChannel));
            return false;
    }

    periodSamples = periodMilli*format.SampleRate/1000;

    uint32_t numOfBytesPerSample = format.BitsPerChannel/8;

    dataPtr.reset(new uint8_t[periodSamples * numOfBytesPerSample * format.Channels],
            std::default_delete<uint8_t[]>());
    samples.DataPtr = dataPtr.get();

    std::string capture("");

    if(!config->GetNumber("disable-real-time-priority-audio", 0))
    {
        backend->setThreadSched(SCHED_FIFO, config->GetNumber("audio-threads-real-time-priority", 61));
    }
    if(inittout_ms>0)
    {
        backend->setInitialTimeout(inittout_ms);
    }
    backend->setFadeTime( FadeMode::OUT,  StreamDirection::OUT, 0);
    AudioError err = backend->openStream(capture, deviceName.c_str(), audioFormat, format.SampleRate, format.Channels,
            periodSamples);

    if (err != AudioError::OK) {
        LOG_ERROR((cply, "%s, AilAudioOutImpl::Impl::Prepare(), openStream() failed, error=%d",
                _getChannelName(channel), static_cast<uint32_t>(err)));
        return false;
    } else {
        LOG_INFO((cply, "%s, AilAudioOutImpl::Impl::Prepare(), openStream() success, device=%s, periods: %d",
                _getChannelName(channel),deviceName.c_str(),periodSamples));
    }

    return true;
}

bool AilAudioOut::Impl::Start()
{
    running = true;

    sampleNumber = 0;
    AudioError err = backend->startStream();
    if (err != AudioError::OK) {
        LOG_ERROR((cply, "%s, AilAudioOutImpl::Impl::Start(), startStream() failed, error=%d", _getChannelName(channel),
                static_cast<uint32_t>(err)));
        return false;
    } else {
        LOG_INFO((cply, "%s, AilAudioOutImpl::Impl::Start(), startStream() SUCCESS", _getChannelName(channel)));
    }
    return true;
}

void AilAudioOut::Impl::Stop()
{
    bool wasRunning = running;
    running = false;

    if(wasRunning) {
        AudioError err = backend->stopStream();
        if (err != AudioError::OK) {
            LOG_ERROR((cply, "%s, stopStream() failed, error=%d", _getChannelName(channel), static_cast<uint32_t>(err)));
        } else {
            LOG_INFO((cply, "%s,  stopStream() SUCCESS", _getChannelName(channel)));
        }

        err = backend->closeStream();
        if (err != AudioError::OK) {
            LOG_ERROR((cply, "%s, closeStream() failed, error=%d", _getChannelName(channel), static_cast<uint32_t>(err)));
        } else {
            LOG_INFO((cply, "%s, closeStream() SUCCESS", _getChannelName(channel)));
        }
        LOG_INFO((cply, "%s audio out stopped", _getChannelName(channel)));
    }
}

void AilAudioOut::Impl::Flush()
{
    if(running) {
        AudioError err = backend->abortStream();
        if (err != AudioError::OK) {
            LOG_ERROR((cply, "%s, abortStream() failed, error=%d", _getChannelName(channel), static_cast<uint32_t>(err)));
        } else {
            LOG_INFO((cply, "%s,  abortStream() SUCCESS", _getChannelName(channel)));
        }

        err = backend->startStream();
        if (err != AudioError::OK) {
            LOG_ERROR((cply, "%s, AilAudioOutImpl::Impl::Start(), startStream() failed, error=%d", _getChannelName(channel),
                    static_cast<uint32_t>(err)));
        } else {
            LOG_INFO((cply, "%s, AilAudioOutImpl::Impl::Start(), startStream() SUCCESS", _getChannelName(channel)));
        }
    }
}

void AilAudioOut::Impl::error(const std::string& data) const
{
    LOG_ERROR((cply, "%s, %s",_getChannelName(channel), data.c_str()));
}
void AilAudioOut::Impl::warning(const std::string& data) const
{
    LOG_WARN((cply, "%s, %s",_getChannelName(channel), data.c_str()));
}
void AilAudioOut::Impl::info(const std::string& data) const
{
    LOG_INFO((cply, "%s, %s",_getChannelName(channel), data.c_str()));
}
void AilAudioOut::Impl::debug(const std::string& data) const
{
    LOGD_DEBUG((cply, "%s, %s",_getChannelName(channel), data.c_str() ));
}
eLogLevel AilAudioOut::Impl::checkLogLevel() const
{
    return eLogLevel::LL_MAX;
}

AudioState AilAudioOut::Impl::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    (void) in;
    const uint32_t numOfBytesPerSample = format.BitsPerChannel/8;
    const int bufferSize = periodSamples *
            (numOfBytesPerSample) * format.Channels;

    samples.Length = bufferSize;
    samples.TimeStamp = sampleNumber;

    source->Read(samples);
    sampleNumber += periodSamples;
    *out = (unsigned char*)samples.DataPtr;
    frames = samples.Length/(numOfBytesPerSample * format.Channels);
    if (verboseLogging) {
        LOGD_DEBUG((cply, "%s Data process - running numOfSamples%zd DataPtr%d", _getChannelName(channel),
                samples.Length, *(uint32_t*)(samples.DataPtr)));
    }

    return AudioState::CONTINUE;
}
void AilAudioOut::Impl::statistics(const StreamStatistics &status)
{
    (void) status;
}
void AilAudioOut::Impl::eostreaming(const AudioError error)
{
    if (error != AudioError::OK) {
        LOG_ERROR((cply, "%s, eostreaming(): Streaming has stopped unexpectedly: %d", _getChannelName(channel),
                (uint32_t)error));
    }
}
} } // namespace adit { namespace carplay
